<?php
/**
* Web Application Component Toolkit
*
* @link http://www.phpwact.org/
*
* @author Wact Development Team
* @link http://www.phpwact.org/team
*
* @copyright Copyright 2006, Jeff Moore
* @license http://opensource.org/licenses/mit-license.php MIT
*/

/**
* Represents an HTML select tag where only a single option can
* be selected
*/
class Wact_View_Form_Select_One extends Wact_View_Form_Element {

    /**
    * A associative array of choices to build the option list with
    * @var array
    */
    protected $choiceList = array();

    /**
    * The object responsible for rendering the option tags
    * @var Wact_View_Form_Select_Option
    */
    protected $optionRenderer;

    /**
    * Current selection
    * @var array
    */
    protected $selection;

    public function __construct($attributes = array(), $binding = NULL) {
        parent::__construct('select', $attributes, $binding);
    }
    
    /**
    * @return array the contents of the value
    */
    public function getSelection() {
        if (is_null($this->selection)) {
            $binding = $this->getAttribute('name');
            $this->selection = $this->transitionParent->getTransition()->getPost($binding);
        }

        return $this->selection;
    }

    /**
    * Sets the choice list. Passed an associative array, the keys become the
    * contents of the option value attributes and the values in the array
    * become the text contents of the option tag e.g.
    * <code>
    * $choices = array(4 => 'red', 5 => 'blue', 6 => 'green');
    * </code>
    * ...becomes...
    * <pre>
    * <select>
    *   <option value="4">red</option>
    *   <option value="5">blue</option>
    *   <option value="6">green</option>
    * </select>
    * </pre>
    * @see setSelection()
    * @param array
    * @return void
    */
    public function setChoices($choiceList) {
        $this->choiceList = $choiceList;
    }

    /**
    * Sets a single option to be displayed as selected. Value
    * should correspond to a key in the array passed to
    * setChoices() e.g.
    * <code>
    * $choices = array(4 => 'red', 5 => 'blue', 6 => 'green');
    * $selection = 5;
    * </code>
    * ...becomes...
    * <pre>
    * <select multiple>
    *   <option value="4">red</option>
    *   <option value="5" selected>blue</option>
    *   <option value="6">green</option>
    * </select>
    * </pre>
    * @see setChoices()
    * @param string option which is selected
    * @return void
    */
    public function setSelection($selection) {
        $this->selection = $selection;
    }

    /**
    * Sets object responsible for rendering the options
    * Supply your own OptionRenderer if the default
    * is too simple
    * @see Wact_View_Form_Select_Option
    * @return void
    */
    public function setOptionRenderer($optionRenderer) {
        $this->optionRenderer = $optionRenderer;
    }

    function hasContents() {
        return TRUE;
    }

    /**
    * Renders the contents of the the select tag, option tags being built by
    * the option handler. Called from with a compiled template render function.
    * @return void
    */
    public function paint() {
        $value = $this->getSelection();

        if (empty($value) || !array_key_exists($value, $this->choiceList)) {
            $value = reset($this->choiceList);
        }

        if (empty($this->optionRenderer)) {
            $this->optionRenderer = new Wact_View_Form_Select_Option();
        }

        foreach($this->choiceList as $key => $contents) {
            if ($key === 0) {
                $key = '0';
            }
            $this->optionRenderer->renderOption($key, $contents, $key == $value);
        }
    }
}

?>