<?php
/**
* Web Application Component Toolkit
*
* @link http://www.phpwact.org/
*
* @author Wact Development Team
* @link http://www.phpwact.org/team
*
* @copyright Copyright 2006, Jeff Moore
* @license http://opensource.org/licenses/mit-license.php MIT
*/

/**
* Represents an HTML select multiple tag where multiple options
* can be selected
*/
class Wact_View_Form_Select_Multiple extends Wact_View_Form_Element {
    /**
    * An associative array of choices to build the option list with
    * @var array
    */
    protected $choiceList = array();

    /**
    * The object responsible for rendering the option tags
    * @var Wact_View_Form_Select_Option
    */
    protected $optionRenderer;
    
    /**
    * Current selection
    * @var array
    */
    protected $selection;

    public function __construct($attributes = array(), $binding = NULL) {
        parent::__construct('select', $attributes, $binding);
    }
    
    public function getSelection() {
        if (is_null($this->selection)) {
            $binding = str_replace('[]', '', $this->getAttribute('name'));
            $this->selection = $this->transitionParent->getTransition()->getPost($binding);
        }

        return $this->selection;
    }
    
    /**
    * Sets the choice list. Passed an associative array, the keys become the
    * contents of the option value attributes and the values in the array
    * become the text contents of the option tag e.g.
    * <code>
    * $choices = array(4 => 'red', 5 => 'blue', 6 => 'green');
    * </code>
    * ...becomes...
    * <pre>
    * <select multiple="multiple">
    *   <option value="4">red</option>
    *   <option value="5">blue</option>
    *   <option value="6">green</option>
    * </select>
    * </pre>
    * @see setSelection()
    * @param array
    * @return void
    */
    function setChoices($choiceList) {
        $this->choiceList = $choiceList;
    }

    /**
    * Sets a list of values to be displayed as selected. These should
    * correspond to the <i>keys</i> of the array passed to setChoices()
    * e.g.
    * <code>
    * $choices = array(4 => 'red', 5=>'blue', 6=>'green');
    * $selections = array(4, 6);
    * </code>
    * ...becomes...
    * <pre>
    * <select multiple="multiple">
    *   <option value="4" selected="selected">red</option>
    *   <option value="5">blue</option>
    *   <option value="6" selected="selected">green</option>
    * </select>
    * </pre>
    * @see setChoices()
    * @param array indexed array of selected options
    * @return void
    */
    public function setSelection($selection) {
        $this->selection = $selection;
    }

    /**
    * Sets object responsible for rendering the options
    * Supply your own OptionRenderer if the default
    * is too simple
    * @see Wact_View_Form_Select_Option
    * @param object
    * @return void
    */
    function setOptionRenderer($optionRenderer) {
        $this->optionRenderer = $optionRenderer;
    }

    function hasContents() {
        return TRUE;
    }

    /**
    * Renders the contents of the the select tag, option tags being built by
    * the option handler. Called from with a compiled template render function.
    * @return void
    */
    function paint() {
        $values = $this->getSelection();

        if (!is_array($values)) {
            $values = array(reset($this->choiceList));
        } else {
            $found = false;
            foreach ($values as $value) {
                if (array_key_exists($value, $this->choiceList)) {
                    $found = true;
                    break;
                }
            }
            if (!$found)
                $values = array(reset($this->choiceList));
        }

        if (empty($this->optionRenderer)) {
            $this->optionRenderer = new Wact_View_Form_Select_Option();
        }

        foreach($this->choiceList as $key => $contents) {
            if ($key === 0) {
                $key = '0';
            }
            $this->optionRenderer->renderOption($key, $contents, in_array($key,$values));
        }
    }
}

?>