<?php
/**
* Web Application Component Toolkit
*
* @link http://www.phpwact.org/
*
* @author Wact Development Team
* @link http://www.phpwact.org/team
*
* @copyright Copyright 2006, Jeff Moore
* @license http://opensource.org/licenses/mit-license.php MIT
*/

/**
* Provides an API for generating member variables for classes
*/
class Wact_Php_Writer_Property {
    /**
    * Name of the property
    * @var string
    */
    protected $name;

    /**
    * Default value
    * @var mixed
    */
    protected $value;

    /**
    * Property visibility
    * @var string public, protected or private
    */
    protected $visibility;

    /**
    * Class constructor
    *
    * @param string $name Name of the property
    * @param mixed $value Default value
    * @param string $visibility public, protected or private
    */
    public function __construct($name, $value = null, $visibility = 'public') {
        if (!preg_match('/^[a-zA-Z_\x7f-\xff][a-zA-Z0-9_\x7f-\xff]*$/', $name)) {
            throw new Wact_Php_Writer_Exception('Invalid Identifier "{Identifier}"', $name);
        }
        $this->name = $name;
        $this->value = $value;
        $this->visibility = $visibility;
    }

    /**
    * Returns the name of the property
    * @return string
    */
    public function getName() {
        return $this->name;
    }

    /**
    * Sets the default value for the property.
    * This can be done using the constructor parameters also.
    * @param mixed $value
    */
    public function setValue($value = null) {
        $this->value = $value;
    }

    /**
    * Sets the visibility of the property.
    * This can be done using the constructor parameters also.
    * @param string $visibility
    */
    public function setVisibility($visibility = 'public') {
        $this->visibility = $visibility;
    }

    /**
    * Returns the declaration code for this property
    * @return string
    */
    public function getDeclaration() {
        $code =ltrim($this->visibility . ' ') . '$' . $this->name;
        if ($this->value !== NULL) {
            $code .= ' = ' .var_export($this->value, TRUE);
        }
        $code .= ';';
        return $code;
    }
}

?>